//https://github.com/urho3d/Urho3D/blob/master/bin/CoreData/Shaders/GLSL/BRDF.glsl
//https://google.github.io/filament/Filament.md.html
//https://schuttejoe.github.io/post/disneybsdf/
//https://github.com/wdas/brdf/blob/main/src/brdfs/disney.brdf

#if defined DEFERRED || defined COMPOSITE
    float schlick_fresnel(float VdotH, float ior, float f0){
        return (ior - f0) * pow5(1.0 - VdotH) + f0;
    }

    float schlick_gaussian_fresnel(float VdotH, float ior, float f0){
        return (ior - f0) * pow(2.0, (-5.55473 * VdotH - 6.98316) * VdotH) + f0;
    }

    float schlick_weight(float u){
        float m = saturate(1.0 - u);
        float m2 = m * m;

        return m * m2 * m2;
    }

    float schlick_ggx(float NdotV, float alpha){
        float a = alpha * alpha;
        float b = NdotV * NdotV;

        return 1.0 / (NdotV + sqrt(a + b - a * b));
    }

    vec3 exact_fresnel(const vec3 n, const vec3 k, float c) {
        const vec3 k2= k * k;
        const vec3 n2k2 = n * n + k2;

        vec3 c2n = (c * 2.0) * n;
        vec3 c2 = vec3(c * c);

        vec3 rs_num = n2k2 - c2n + c2;
        vec3 rs_den = n2k2 + c2n + c2;

        vec3 rs = rs_num / rs_den;

        vec3 rp_num = n2k2 * c2 - c2n + 1;
        vec3 rp_den = n2k2 * c2 + c2n + 1;

        vec3 rp = rp_num / rp_den;

        return saturate(0.5 * (rs + rp));
    }

    vec3 f0_to_ior(vec3 f0){
        return (1.0 + f0) / (1.0 - f0);
    }

    vec3 fresnel(specular s, vec3 albedo, float theta, bool water){
        if(water){
            return vec3(schlick_gaussian_fresnel(theta, s.ior, s.f0));
        } else if(s.conductor >= 230 && s.conductor <= 237){
            #if SPEC_FORMAT == 0
                return exact_fresnel(s.n, s.k, theta) * sqrt(albedo);
            #else
                return exact_fresnel(f0_to_ior(albedo) * s.ior, vec3(0.0), theta);
            #endif
        } else {
            return exact_fresnel(f0_to_ior(albedo) * s.ior, vec3(0.0), theta);
        }
    }

    float smith_geometry_g1(float NdotV, float roughness){
        return (2.0 * NdotV) / (NdotV + sqrt(roughness + (1.0 - roughness) * (NdotV * NdotV)));
    }

    float smith_geometry_g2(float NdotL, float NdotV, float roughness){
        float gl = smith_geometry_g1(NdotL, roughness);
        float gv = smith_geometry_g1(NdotV, roughness);

        return gl * gv;
    }

    float exact_correlated_g2(float NdotL, float NdotV, float roughness) {
        float x = 2.0 * NdotL * NdotV;
        float y = 1.0 - roughness;

        return x / (NdotV * sqrt(roughness + y * (NdotL * NdotL)) + NdotL * sqrt(roughness + y * (NdotV * NdotV)));
    }

    float ggx_g2(float NdotL, float NdotV, float roughness){
        float g0 = (2.0 * NdotV) / (NdotV + sqrt(roughness + (1.0 - roughness) * (NdotV * NdotV)));
        float g1 = (2.0 * NdotL) / (NdotL + sqrt(roughness + (1.0 - roughness) * (NdotL * NdotL)));

        return g0 * g1;
    }

    float d_ggx(float NdotH, float roughness){
        float tmp = (NdotH * roughness - NdotH) * NdotH + 1.0;
        return roughness / (tmp * tmp) * rpi;
    }

    float smith_ggx_correlated(float NoV, float NoL, float a){
        float a2 = a * a;
        float GGXL = NoV * sqrt((-NoL * a2 + NoL) * NoL + a2);
        float GGXV = NoL * sqrt((-NoV * a2 + NoV) * NoV + a2);
        return 0.5 / (GGXV + GGXL);
    }

    float disney_diffuse(float NdotL, float NdotV, float LdotH, float roughness, float ior){
        float l0 = schlick_weight(NdotL);
        float l1 = schlick_weight(NdotV);

        float f0 = 0.5 + 2.0 * roughness * NdotL * NdotL;
        float b = f0 * (l0 + l1 + l0 * l1 * (f0 - 1.0));

        return rpi * (b * (1.0 - 0.5 * l0) + (1.0 - 0.5 * l1));
    }

    vec3 hammon_diffuse(vec3 albedo, float NdotL, float NdotV, float NdotH, float LdotV, float roughness){
        float facing = 0.5 + 0.5 * LdotV;
    
        float sample_rough = facing * (0.9 - 0.4 * facing) * (0.5 + NdotH / NdotH);
        float sample_smooth = 1.05 * (1.0 - pow5(1.0 - NdotL)) * (1.0 - pow5(1.0 - NdotV));

        float single = mix(sample_smooth, sample_rough, roughness) * rpi;
        float multi = 0.1159 * roughness;

        return single + albedo * multi;
    }

    float NdotH_squared(float radiusTan, float NoL, float NoV, float VoL){
        float radiusCos = inversesqrt(1.0 + radiusTan * radiusTan);
        
        float RoL = 2.0 * NoL * NoV - VoL;
        if (RoL >= radiusCos)
            return 1.0;

        float rOverLengthT = radiusCos * radiusTan * inversesqrt(1.0 - RoL * RoL);
        float NoTr = rOverLengthT * (NoV - RoL * NoL);
        float VoTr = rOverLengthT * (2.0 * NoV * NoV - 1.0 - RoL * VoL);

        float triple = sqrt(clamp(1.0 - NoL * NoL - NoV * NoV - VoL * VoL + 2.0 * NoL * NoV * VoL,0.,1.));
        
        float NoBr = rOverLengthT * triple, VoBr = rOverLengthT * (2.0 * triple * NoV);
        float NoLVTr = NoL * radiusCos + NoV + NoTr, VoLVTr = VoL * radiusCos + 1.0 + VoTr;
        float p = NoBr * VoLVTr, q = NoLVTr * VoLVTr, s = VoBr * NoLVTr;    
        float xNum = q * (-0.5 * p + 0.25 * VoBr * NoLVTr);
        float xDenom = p * p + s * ((s - 2.0 * p)) + NoLVTr * ((NoL * radiusCos + NoV) * VoLVTr * VoLVTr + 
                    q * (-0.5 * (VoLVTr + VoL * radiusCos) - 0.5));
        float twoX1 = 2.0 * xNum / (xDenom * xDenom + xNum * xNum);
        float sinTheta = twoX1 * xDenom;
        float cosTheta = 1.0 - twoX1 * xNum;
        NoTr = cosTheta * NoTr + sinTheta * NoBr;
        VoTr = cosTheta * VoTr + sinTheta * VoBr;
        
        float newNoL = NoL * radiusCos + NoTr;
        float newVoL = VoL * radiusCos + VoTr;
        float NoH = NoV + newNoL;
        float HoH = 2.0 * newVoL + 2.0;
        return max(0.0, NoH * NoH / HoH);
    }

    vec3 ggx_hightligh(specular s, materials m, vec3 albedo, vec3 normal, vec3 vector, vec3 direction){
        atmosphere_constant ac = atmosphere_s();

        #if defined DEFERRED7
            normal = m.grass ? vec3(0.0, 1.0, 0.0) : normal;
        #endif

        vec3 H = normalize(direction - vector);

        float radius = sunAngle < 0.5 ? ac.sun_angular_radius : ac.moon_angular_radius;

        float NdotL = dot(normal, direction);
        float NdotV = dot(normal, -vector);
        float LdotV = dot(direction, -vector);
        float LdotH = dot(direction, H);
        float NdotH = NdotH_squared(radius, NdotL, NdotV, LdotV);

        float d = d_ggx(abs(NdotH), s.roughness);
        float v = exact_correlated_g2(abs(NdotL), abs(NdotV), s.roughness);
        vec3 f = fresnel(s, albedo, abs(LdotH), m.water);

        return f * d * v / (4.0 * NdotL * NdotV + 1e-7);
    }
#endif